﻿/*
VERSION: 	1.5
1.5		getLoopingViewArea()  changed to accomodate movieClip patterns
1.4		getWrappedCoords()  when isLooping is false,  wrapBounds is handled differently to allow the same bounds to be used for both looping and non-looping
1.3		shrinkRectCenter()  newSize_rect can optionally be an object with only width+height.  (it'll be converted internally into a Rectangle)
1.2		refactored almost every function:  Further isolated distinct functionality.  Exposed the output data as much as possible.  (no more mysterious object blobs being passed around)
1.1		getWrappedCoords()		This function's non-looping behavior now allows diagonally sliding along the edges.

LIST OF FUNCTIONS: 
	generatePattern( original_pic )
	rotatePoint( input_p, apply_clockwise_radian )
	getWrapBounds( original_pic )
	getWrappedCoords( input_coords, wrapBounds, isLooping )
	getCenterMatrix( original_pic )
	getLoopingViewArea( pattern_pic )
	shrinkRectCenter( old_rect, newSize_rect )
	render( pattern_pic, pos_topLeft_p, radian, center_mat, crop_rect, crop_mat, zoomScale, eraseColor, output_pic )
	
NOTE: 
	The position coords determine the top-left corner of the rendered image.
	If you want the position to appear to be centered,  you can use the "center_mat" provided by getCenterMatrix()  to offset your coordinates before sending them to getWrappedCoords()... or you can use your own arbitrary offset to change the apparent origin.
	
	// wrap the coords
		// // Display the player position in the center,  via offset
		playerPos_p.x += center_mat.tx;		// 0 => -512
		playerPos_p.y += center_mat.ty;
		var wrappedCoords_p = loopImageHelper.getWrappedCoords( playerPos_p, wrapBounds, isLooping );		// looping: 0 ... 1024   no-looping: -512 ... +512
		// display the wrapped coords
		var renderCoords_p = wrappedCoords_p;
		playerPos_p = wrappedCoords_p.clone();
		playerPos_p.x -= center_mat.tx;		// -512 => 0		// looping: 512 ... 1536   no-looping: 0 ... 1024
		playerPos_p.y -= center_mat.ty;
		if( playerPos_p.x > original_pic.width )		playerPos_p.x %= original_pic.width;		// looping + un-center can result in coords greater than the original image bounds
		if( playerPos_p.y > original_pic.height )		playerPos_p.y %= original_pic.height;
		
	
USAGE:
	#include "functions/loopImageHelper.as"
	var BitmapData = flash.display.BitmapData;
	var Point = flash.geom.Point;
	var zoomScale = 0.5;		// anything smaller than 1 will reveal the wrap edges
	var angle = 0;
	var radian = angle * 2*Math.PI / 360;
	// get a world-map / rolling-image
	var original_pic = BitmapData.loadBitmap( "test.png" );
	// make a loop-able pattern
	var pattern_pic = loopImageHelper.generatePattern( original_pic );
	// keeps coords within the image
	var coordsAfterVel_p = new Point( 0,0 );
	var vel_p = new Point( 0, -10 );
	var rotatedVel_p = loopImageHelper.rotatePoint( vel_p, radian );
	coordsAfterVel_p.add( rotatedVel_p );
	var isLooping = true;
	var wrapBounds = loopImageHelper.getWrapBounds( original_pic );
	var renderCoords_p = loopImageHelper.getWrappedCoords( coordsAfterVel_p, wrapBounds, isLooping );
	// determine the rotation-point  (typically at the center)
	var center_mat = loopImageHelper.getCenterMatrix( original_pic );
	// get a view rectangle
	var cropData = loopImageHelper.getLoopingViewArea( pattern_pic );
	var crop_rect = cropData.crop_rect;
	var crop_mat = cropData.crop_mat;
	// shrink the view-area
	var crop_rect = loopImageHelper.shrinkRectCenter( crop_rect, {width:300, height: 300} );
	// apply cropYScale  (change the rotation-point of the output image, by cropping the render-image after rotation)
	var cropYScale = 0.6;
	crop_rect.height *= cropYScale;
	// render
	var eraseColor = null;
	var rendered_pic = loopImageHelper.render( pattern_pic, renderCoords_p, radian, center_mat, crop_rect, crop_mat, zoomScale, eraseColor, render_pic );
	// display the render
	var test_mc = this.createEmptyMovieClip( "test_mc", 0 );
	// test_mc._xscale = test_mc._yscale = 75;
	test_mc.attachBitmap( rendered_pic, 0 );
	
	// animate map-coords + display-angle
	onEnterFrame = function(){
		renderCoords_p.y -= 10;		// change coords on the map
		radian += 0.01;				// spin the output image
		renderCoords_p = loopImageHelper.getWrappedCoords( renderCoords_p, wrapBounds, isLooping );
		loopImageHelper.render( pattern_pic, renderCoords_p, radian, center_mat, crop_rect, crop_mat, zoomScale, eraseColor, rendered_pic );
	}// loop()
	
	

DETAILS ABOUT FUNCTIONS: 
	generatePattern( original_pic );
		// generate pattern
		input:	bitmap
		output	pattern_pic		(bitmap)
	
	
	rotatePoint( input_p, apply_clockwise_radian );
		// Adds rotation to a Point  (a Point object is technically a vector)
		vector + angle  =  rotated vector  (clockwise)
		
		input:	local_vector, clockwise_radian
		output:	rotated_vector
		
		This enables relative movement like so: 
			Take local velocity
			Apply global rotation to local velocity
			This results in a global velocity
			Add this global velocity to the global position
	
	
	getWrapBounds( original_pic );
		// generate wrap-boundaries
		input:	bitmap
		output	wrapBounds		{ xMin,yMin,xMax,yMax }
	
	
	getWrappedCoords( input_coords, wrapBounds, isLooping );
		// coords + wrap-boundaries + doWrap + direction-vector = new coords
		input:	in_coords			{ x,y }		(attempt these coords)
						wrapBounds		{ xMin,yMin,xMax,yMax }
						isLooping			[optional]		true/false
		output:	Point:	x,y
	
	
	getCenterMatrix( original_pic );
		// get the center-coords of an image,  as a matrix
		input:	bitmap
		output:	center_mat
	
	
	getLoopingViewArea( pattern_pic );
		// Determine the maximum view-size that will still hide the pattern-looping
		input:		BitmapData		(pattern bitmap)
		output: 	crop_rect			Rectangle			(max size that will hide wrapping)
							crop_mat			Matrix				(offset to re-positon the cropped image into the top-left of the render bitmap)
	
	
	shrinkRectCenter( old_rect, newSize_rect );
		// shrink the view-area
		Resizes a rectangle and to the specified size... relative to its center
		input:		Rectangle		(old_rect)
							Rectangle		(newSize_rect)
		output:		Rectangle		(rectangle scaled relative to its center)
	
	
	render( pattern_pic, pos_topLeft_p, radian, center_mat, crop_rect, crop_mat, zoomScale, eraseColor, output_pic );
		// coords + pattern = output pic
		input:	pattern_pic			bitmap					(pattern bitmap)
						pos_topLeft_p		x,y							(view-point top-left coords)
						radian					(view-point angle)		(the viewing direction.  Clockwise.  0 = North)
						center_mat			(Matrix)				(coords stored as a matrix which controls the rotation-point)
						crop_rect				(Rectangle)			(used to crop the rendered bitmap)
						crop_mat				(Matrix)				(used to offset the rendered bitmap into the cropped area)
						zoomScale				(Scalar)				(zoom-in / zoom-out on the rotation-point)
						eraseColor			(Number)				(the color used for the edges of the rendered image when wrapping is disabled)
						[render_pic]		(BitmapData)		[optional]		(output bitmap canvas to draw the results to)
		output:	render_pic			(BitmapData)		Resulting bitmap.  This is the render_pic (if provided)  or a newly generated bitmap (if no render_pic is provided)

*/
var loopImageHelper = {
	/*****************************************
	*****************************************/
	
	
	
	/*****************************************
	generate pattern
	input:	bitmap
	output	pattern_pic		(bitmap)
	*****************************************/
	generatePattern: function( original_pic ){
		// convenient shortcuts
		var Point = flash.geom.Point;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		if( original_pic instanceof BitmapData === false )		throw new Error("generatePattern() requires BitmapData input");
		// internal parameters
		var edgeTransparent = false;
		
		
		
		// prepare the pattern for wrapping
		var ww = original_pic.width *2;
		var hh = original_pic.height *2;
		var pattern_pic = new BitmapData( ww,hh, edgeTransparent,0 );
		
		// copy the original into the prepped image
		var copy_rect = new Rectangle( 0,0, original_pic.width, original_pic.height );
		var paste_p = new Point( 0, 0 );
		pattern_pic.copyPixels( original_pic, copy_rect, paste_p );
		
		
		
		// add loop-copies to the prepped image
		// // create right buffer
		// // // copy left part
		var xx = 0;
		var yy = 0;
		var ww = original_pic.width;
		var hh = pattern_pic.height;
		var copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the right
		var xx = original_pic.width;
		var yy = 0;
		var paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		// // create bottom loop-copies
		// // // copy top half
		var xx = 0;
		var yy = 0;
		var ww = pattern_pic.width;
		var hh = original_pic.height;
		var copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the bottom half
		var xx = 0;
		var yy = original_pic.height;
		var paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		
		var output = pattern_pic;
		return output;
	}, //generatePattern()
	
	
	
	/*****************************************
	Adds rotation to a Point  (a Point object is technically a vector)
	vector + angle  =  rotated vector  (clockwise)
	
	input:	local_vector, clockwise_radian
	output:	rotated_vector
	
	This enables relative movement like so: 
		Take local velocity
		Apply global rotation to local velocity
		This results in a global velocity
		Add this global velocity to the global position
	*****************************************/
	rotatePoint: function( input_p, apply_clockwise_radian ){
		var Matrix = flash.geom.Matrix;
		var Point = flash.geom.Point;
		
		// Matrix technique
		var work_mat = new Matrix( 1,0,0,1, -input_p.x, -input_p.y );
		work_mat.rotate( apply_clockwise_radian);
		var output = new Point( work_mat.tx, work_mat.ty );
		return output;
	}, // rotatePoint()
	
	
	
	/*****************************************
	generate wrap-boundaries
	input:	bitmap
	output	wrapBounds		{ xMin,yMin,xMax,yMax }
	*****************************************/
	getWrapBounds: function( original_pic ){
		var wrapBounds = {
			xMin: 0,
			yMin: 0,
			xMax: original_pic.width,
			yMax: original_pic.height
		};
		wrapBounds.toString = function(){
			return "min: ("+this.xMin+", "+this.yMin+")  max: ("+this.xMax+", "+this.yMax+")";
		}
		return wrapBounds;
	}, // getWrapBounds()
	
	
	
	/*****************************************
	coords + wrap-boundaries + doWrap + direction-vector = new coords
	input:	in_coords			{ x,y }		(attempt these coords)
					wrapBounds		{ xMin,yMin,xMax,yMax }
					isLooping			[optional]		true/false
	output:	Point:	x,y
	*****************************************/
	getWrappedCoords: function( input_coords, wrapBounds, isLooping ){
		// convenient shortcuts
		var Point = flash.geom.Point;
		
		// resolve passed parameters
		var hasCoords = !( input_coords.x === undefined  ||  input_coords.y === undefined );
		if( !hasCoords )		throw new Error("getWrappedCoords() did not receive coordinates");
		var hasNumCoords = !( isNaN(input_coords.x)  ||  isNaN(input_coords.y) );
		if( !hasNumCoords )		throw new Error("getWrappedCoords() received invalid non-numeric coordinates");
		var hasWrapBounds = !( wrapBounds.xMin === undefined || wrapBounds.yMin === undefined || wrapBounds.xMax === undefined || wrapBounds.yMax === undefined);
		if( !hasWrapBounds )		throw new Error("getWrappedCoords() did not receive wrapBounds");
		var hasNumWrapBounds = !( isNaN(wrapBounds.xMin) || isNaN(wrapBounds.yMin) || isNaN(wrapBounds.xMax) || isNaN(wrapBounds.yMax) );
		if( !hasNumWrapBounds )		throw new Error("getWrappedCoords() received invalid non-numeric wrapBounds");
		var hasLoopingParam = !( isLooping === undefined );
		if( !hasLoopingParam )		var isLooping = true;
		
		
		// 
		var output = new Point( input_coords.x, input_coords.y );
		
		
		if( isLooping )
		{// if:  coordinates should wrap-around
			while( output.x < 0 )		output.x += wrapBounds.xMax;
			while( output.y < 0 )		output.y += wrapBounds.yMax;
			if( output.x >= wrapBounds.xMax )		output.x %= wrapBounds.xMax;
			if( output.y >= wrapBounds.yMax )		output.y %= wrapBounds.yMax;
		}// if:  coordinates should wrap-around
		
		else
		
		// This version allows for diagonally sliding along the edges.
		{// if:  coordinates should stop and collide at edges
			var width = wrapBounds.xMax - wrapBounds.xMin;
			var height = wrapBounds.yMax - wrapBounds.yMin;
			var halfWidth = width/2;
			var halfHeight = height/2;
			if( output.x < wrapBounds.xMin-halfWidth)		output.x = wrapBounds.xMin-halfWidth;
			else if( output.x > wrapBounds.xMax-halfWidth)		output.x = wrapBounds.xMax-halfWidth;
			if( output.y < wrapBounds.yMin-halfHeight)		output.y = wrapBounds.yMin-halfHeight;
			else if( output.y > wrapBounds.yMax-halfHeight)		output.y = wrapBounds.yMax-halfHeight;
		}// if:  coordinates should stop and collide at edges
		
		
		return output;
	}, // getWrappedCoords()
	
	
	
	/*****************************************
	get the center-coords of an image,  as a matrix
	input:	bitmap
	output:	center_mat
	*****************************************/
	getCenterMatrix: function( original_pic ){
		var Matrix = flash.geom.Matrix;
		var center_mat = 			new Matrix( 1,0,0,1, -(original_pic.width/2), -(original_pic.height/2) );		// half width  of a single image iteration  (exclude duplicate buffer on the right)
		var center_neg_mat = 	new Matrix( 1,0,0,1,  (original_pic.width/2),  (original_pic.height/2) );		// half height of a single image iteration  (exclude duplicate buffer on the bottom)
		return center_mat;
	}, // getCenterMatrix()
	
	
	
	/*****************************************
	Determine the maximum view-size that will still hide the pattern-looping
	input:		BitmapData		(pattern bitmap)
	output: 	crop_rect			Rectangle			(max size that will hide wrapping)
						crop_mat			Matrix				(offset to re-positon the cropped image into the top-left of the render bitmap)
	*****************************************/
	getLoopingViewArea: function( pattern_pic ){
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		var hasPattern = ( pattern_pic instanceof BitmapData )  ||  ( pattern_pic instanceof MovieClip );
		if( !hasPattern )		throw new Error( "makeRenderSurface() did not receive a pattern BitmapData" );
		
		// internal parameters
		var edgeOpaque = true;
		var halfImageScale = 4;		// half of the original bitmap image  (excluding the cloned buffers on the right and bottom)
		
		// Create the render-display image.
		// figure out the maximum safe crop area,  so that no out-of-image areas will ever be seen
		var patternWidth = pattern_pic.width || pattern_pic._width;
		var patternHeight = pattern_pic.height || pattern_pic._height;
		var aspectRatio = patternWidth / patternHeight;
		//var rad_45 = 0.78539816339744830961566084581988;		// 45 degrees, in radians  (where the rectangle touches the safe-circle of a square image)
		//var x_scalar = Math.cos(rad_45);
		//var y_scalar = Math.sin(rad_45);	// omitted because, at 45 degrees, sin() and cos() have the same value
		var x_scalar = 0.707106781186548;		// hard-code the resulting value because it will never change
		var y_scalar = x_scalar * aspectRatio;		// x_scalar is used because, at 45 degrees, sin() and cos() have the same value
		var crop_x = Math.round((patternWidth/halfImageScale)			*(1-x_scalar));			// left offset	(half of a single image  (exclude duplicate buffer)
		var crop_y = Math.round((patternHeight/halfImageScale)			*(1-y_scalar));			// top offset		(half of a single image  (exclude duplicate buffer)
		var crop_width = Math.round((patternWidth/halfImageScale		*x_scalar) *2);
		var crop_height = Math.round((patternHeight/halfImageScale	*y_scalar) *2);
		var crop_mat = new Matrix( 1,0,0,1, -crop_x, -crop_y );
		// var crop_pic = new BitmapData(crop_width, crop_height, !edgeOpaque, 0);
		// crop_pic = new BitmapData( patternWidth, patternHeight, !edgeOpaque, 0);		// used to test clipRect  (without clipRect, a very large image is displayed)
		var crop_rect = new Rectangle( 0,0, crop_width,crop_height  );
		
		/*	
		// crop-down the view if the pattern is too big to display all at once		(BUT... large images fail to load before this ever gets used, so I can't test any of it)
		if( crop_width > 2880  ||  crop_height > 2880 )
		{// pic is too big  =>  crop down the view
			// aspectRatio:		< 1.0 is portrait		> 1.0 is landscape
			if( aspectRatio > 1 )
			{// if:  landscape
				var newWidth = 2880;
				var newHeight = 2880 / aspectRatio;
			}// if:  landscape
			else
			{// if:  portrait
				var newWidth = 2880 / aspectRatio;
				var newHeight = 2880;
			}// if:  portrait
			trace("this.shrinkRectCenter: "+this.shrinkRectCenter);
			newSize_r = new Rectangle( 0,0,  newWidth, newHeight  );
			var new_crop_rect = this.shrinkRectCenter( crop_rect, newSize_r );
			var xDiff = crop_rect.width - new_crop_rect.width;
			var yDiff = crop_rect.height - new_crop_rect.height;
			crop_rect = new_crop_rect;
			//  the x, y of crop_rect is ingored and never used
			// not certain whether this is necessary,  but it probably should be adjusted when the size changes
			crop_mat.tx -= Math.floor( xDiff / 2 );
			crop_mat.ty -= Math.floor( yDiff / 2 );
		}// pic is too big  =>  crop down the view
		*/
		
		
		
		return {
			crop_mat: crop_mat, 
			crop_rect: crop_rect
		};
	}, // getLoopingViewArea()
	
	

	/*****************************************
	// shrink the view-area
	Resizes a rectangle and to the specified size... relative to its center
	input:		Rectangle		(old_rect)
						Rectangle		(newSize_rect)
	output:		Rectangle		(rectangle scaled relative to its center)
	*****************************************/
	shrinkRectCenter: function( old_rect, newSize_rect ){
		var Rectangle = flash.geom.Rectangle;
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		if( isNaN(old_rect.width) )					throw new Error("ERROR:  shrinkRectCenter()  input_rect does not have a 'width'");
		if( isNaN(old_rect.height) )				throw new Error("ERROR:  shrinkRectCenter()  input_rect does not have a 'height'");
		if( isNaN(newSize_rect.width) )			throw new Error("ERROR:  shrinkRectCenter()  newSize_rect does not have a 'width'");
		if( isNaN(newSize_rect.height) )		throw new Error("ERROR:  shrinkRectCenter()  newSize_rect does not have a 'height'");
		
		var wDiff = old_rect.width - newSize_rect.width;
		var hDiff = old_rect.height - newSize_rect.height;
		var wHalf = Math.floor( wDiff / 2 );
		var hHalf = Math.floor( hDiff / 2 );
		
		if( !newSize_rect.clone )		newSize_rect = new Rectangle(  newSize_rect.x||0, newSize_rect.y||0, newSize_rect.width||1, newSize_rect.height||1  );
		var output_rect = newSize_rect.clone();
		output_rect.x = old_rect.x + wHalf;
		output_rect.y = old_rect.y + hHalf;
		return output_rect;
	}, // shrinkRectCenter()
	
	
	
	/*****************************************
	coords + pattern = output pic
	input:	pattern_pic			bitmap					(pattern bitmap)
					pos_topLeft_p		x,y							(view-point top-left coords)
					radian					(view-point angle)		(the viewing direction.  Clockwise.  0 = North)
					center_mat			(Matrix)				(coords stored as a matrix which controls the rotation-point)
					crop_rect				(Rectangle)			(used to crop the rendered bitmap)
					crop_mat				(Matrix)				(used to offset the rendered bitmap into the cropped area)
					zoomScale				(Scalar)				(zoom-in / zoom-out on the rotation-point)
					eraseColor			(Number)				(the color used for the edges of the rendered image when wrapping is disabled)
					[render_pic]		(BitmapData)		[optional]		(output bitmap canvas to draw the results to)
	output:	render_pic			(BitmapData)		Resulting bitmap.  This is the render_pic (if provided)  or a newly generated bitmap (if no render_pic is provided)
	*****************************************/
	render: function( pattern_pic, pos_topLeft_p, radian, center_mat, crop_rect, crop_mat, zoomScale, eraseColor, output_pic ){
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		var hasPattern = ( pattern_pic instanceof BitmapData )  ||  ( pattern_pic instanceof MovieClip );
		if( !hasPattern )		throw new Error( "render() did not receive a pattern that is BitmapData or MovieClip" );
		var hasCoords = !( pos_topLeft_p.x === undefined  ||  pos_topLeft_p.y === undefined );
		if( !hasCoords )		throw new Error("render() did not receive position coordinates");
		var hasNumCoords = !( isNaN(pos_topLeft_p.x)  ||  isNaN(pos_topLeft_p.y) );
		if( !hasNumCoords )		throw new Error("render() received invalid non-numeric position coordinates");
		var hasRadian = !isNaN( radian );
		if( !hasRadian )			throw new Error("render()  The radian angle is either missing or invalid");
		var has_center_mat = (center_mat instanceof Matrix);
		if( !has_center_mat )		throw new Error( "render()  is missing a center-offset Matrix" );
		var has_crop_rect = (crop_rect instanceof Rectangle);
		if( !has_crop_rect )		throw new Error( "render()  is missing a cropping Rectangle" );
		var has_crop_mat = (crop_mat instanceof Matrix);
		if( !has_crop_mat )		throw new Error( "render()  is missing a crop-offset Matrix" );
		
		// create data
		if( isNaN(zoomScale) )		var zoomScale = 1;
		var position_neg_mat =		new Matrix( 1,0,0,1, -pos_topLeft_p.x, -pos_topLeft_p.y );
		var center_neg_mat =			new Matrix( 1, 0, 0, 1, -center_mat.tx, -center_mat.ty );
		var angle_mat =						new Matrix();				angle_mat.rotate( radian );
		if( !output_pic )					var output_pic = new BitmapData( crop_rect.width, crop_rect.height, false, 0 );
		
		// apply position and rotation
		var display_mat = new Matrix();
		// display_mat.identity();
		display_mat.concat( center_mat );				// start at -5,-5		(to rotate around the center)
		display_mat.concat( position_neg_mat );	// add the position coords
		display_mat.concat( angle_mat );				// add the angle		(to rotate)
		display_mat.scale( zoomScale, zoomScale );
			var center_neg_mat_scaled = center_neg_mat.clone();
			center_neg_mat_scaled.scale( zoomScale, zoomScale );
		display_mat.concat( center_neg_mat );		// shift +5,+5			(to display the center at the center)
		
		display_mat.concat( crop_mat );					// offset to top-left of crop area
		
		if( eraseColor !== undefined ){
			var full_rect = new Rectangle( 0,0, output_pic.width,output_pic.height );
			output_pic.fillRect( full_rect, eraseColor );
		}
		
		// use "clipRect" to reduce drawing work
		// // If clipRect has an offset (such as when it's smaller than the maximum width or height),  Then position the rendered result in the top-left corner of the output-bitmap.
		if( clipRect.x > 0 || clipRect.y > 0 ){
			display_mat.translate( -clipRect.x, -clipRect.y );
			var clipRect = new Rectangle( 0,0, clipRect.width, clipRect.height );
		}
		// output_pic.draw( pattern_pic, display_mat );		// the size of output_pic will automatically crop down to the correct width and height
		var interpolation = true;
		output_pic.draw( pattern_pic, display_mat, null, null, clipRect, interpolation );		// the size of output_pic will automatically crop down to the correct width and height
		
		return output_pic;
	}, // render()
	
	
	
	null: null
}// loopImageHelper {}